const Product = require('../models/product')
const formidable = require('formidable')
const _ = require('lodash')
const fs = require('fs')
const category = require('../models/category')


exports.getProductById = (req, res, next, id) => {
    Product.findById(id)
        .populate('category')
        .exec((err, product) => {
            if (err) {
                return res.status(400).json({
                    message: 'Product not found!'
                })
            }
            req.product = product
            next()
        })
}

exports.createProduct = (req, res) => {
    const { name, description, price, category, stock } = req.body

    if (
        !name ||
        !description ||
        !price ||
        !stock
    ) {
        return res.status(400).json({
            message: 'Please include all fields!'
        })
    }

    //TODO: restrictions on field
    const product = new Product(req.body)
    //save to the db
    product.save((err, product) => {
        if (err) {
            console.error(err)
            return res.status(400).json({
                message: 'Saving product to db failed'
            })
        }
        res.json(product)
    })
}

exports.uploadImage = (req, res) => {
    const product = req.product
    if (!req.files) {
        return res.status(400).json({ message: "Can't get Image!" })
    }

    const fileArray = req.files
    fileArray.forEach(file => {
        console.log(file);
        product.photos.push(file.filename)
    });
    product.save((err, product) => {
        if (err) {
            console.error(err)
            return res.status(400).json({
                message: 'Saving product to db failed'
            })
        }
        res.json(product)
    })
}

exports.deleteProductImage = (req, res) => {
    const product = req.product
    const image = req.body.imageName
    const fileArray = product.photos
    fileArray.forEach((item, index) => {
        if (item === image) {
            fileArray.splice(index, 1)
            deletePhoto(item)
        }
    })
    product.photos = fileArray
    product.save((err, product) => {
        if (err) {
            console.error(err)
            return res.status(400).json({
                message: 'Saving product to db failed'
            })
        }
        res.json(product)
    })
}

exports.getProduct = (req, res) => {
    return res.json(req.product)
}

exports.photo = (req, res, next) => {
    if (req.product.photo.data) {
        res.set('Content-Type', req.product.photo.contentType)
        return res.send(req.product.photo.data)
    }
    next()
}

exports.updateProduct = (req, res) => {
    //updation code
    let product = req.product
    product = _.extend(product, req.body)

    //save to the db
    product.save((err, product) => {
        if (err) {
            console.log(err);
            return res.status(400).json({
                message: 'Update t-shirt to db failed'
            })
        }
        res.json(product)
    })

}

exports.deleteProduct = (req, res) => {
    let product = req.product
    product.remove((err, deletedProduct) => {
        if (err) {
            return res.status(400).json({
                message: 'Failed to delete the product!'
            })
        }
        res.json({
            message: 'Deletion is success!',
            deletedProduct
        })
    })
}

//product listing
exports.getAllProducts = (req, res) => {
    // let limit = req.query.limit ? parseInt(req.query.limit) : 8
    let sortBy = req.query.sortBy ? req.query.sortBy : '_id'

    Product.find()
        .select('-photo')
        .populate('category')
        .sort([[sortBy, 'asc']])
        // .limit(limit)
        .exec((err, products) => {
            if (err) {
                return res.status(400).json({
                    error: 'No product found!'
                })
            }
            res.json(products)
        })
}

exports.getPopularProducts = (req, res) => {
    //get favorite products
    let limit = req.body.limit ? parseInt(req.body.limit) : 8
    Product.find({ isPopular: true })
        .limit(limit)
        .exec((err, products) => {
            if (err) {
                return res.status(400).json({ message: 'Unable to fetch favorite products' })
            }
            res.send(products)
        })
}

exports.updateStock = (req, res, next) => {
    let myOperations = req.body.order.products.map(prod => {
        return {
            updateOne: {
                filter: { _id: prod._id },
                update: { $inc: { stock: -prod.count, sold: +prod.count } }
            }
        }
    })

    Product.bulkWrite(myOperations, {}, (err, products) => {
        if (err) {
            return res.status(400).json({
                error: 'Bulk operation failed!'
            })
        }
        next()
    })
}

exports.getAllUniqueCategories = (req, res) => {
    Product.distinct('category', {}, (err, category) => {
        if (err) {
            return res.status(400).json({
                error: 'No category found!'
            })
        }
        res.json(category)
    })
}

deletePhoto = (name) => {
    console.log(name);
    fs.unlink(__basedir + '/uploads/' + name, (err) => {
        if (err) {
            console.log(err);
        };
        // console.log("data not found");
    });
}

exports.filterOptions = (req, res) => {
    Product.find()
        .select('-_id price productColor productSize category brandName')
        .populate('category', 'name')
        .exec((err, options) => {
            if (err || !options) {
                return res.status(404).json({ message: 'No Options Found!' })
            }
            const brands = []
            const optionData = []
            const colors = []
            const sizes = []
            let maxPrice = 0
            const categories = []
            // console.log(options);
            options.forEach((option) => {
                option.productColor.forEach((o) => {
                    colors.push(o)
                })
                // option.brandName.forEach((b) => {
                if (option.brandName) {
                    brands.push(option.brandName)
                }
                // })
                option.productSize.forEach((s) => {
                    sizes.push(s)
                })
                categories.push(option.category)
                if (maxPrice < option.price) {
                    maxPrice = option.price
                }
            })
            optionData.push(_.uniq(colors))
            optionData.push(_.uniq(brands))
            optionData.push(_.uniq(sizes))
            optionData.push(_.uniq(categories))
            res.json({ optionData, maxPrice })
        })
}


exports.filterProducts = (req, res) => {

    let { name, priceRange, brands, ratings, discount, size, color, category, sortBy } = req.body

    const productName = name ? name : ''
    const minPrice = priceRange ? priceRange[0] : 0
    const maxPrice = priceRange ? priceRange[1] : 1000


    let limit = req.body.limit ? parseInt(req.body.limit) : 40
    let shortBy = ['price', 'asc']
    if (sortBy) {
        switch (sortBy) {
            case 'popular': {
                shortBy = ["isPopular", 'asc']
            }
                break;
            case 'htl': {
                shortBy = ["price", 'desc']
            }
                break;
            case 'lth': {
                shortBy = ["price", 'asc']
            }
                break;
            case 'newest': {
                shortBy = ["createdAt", 'asc']
            }
                break;

        }
    }
    const filterQuery = {
        $and: [{ price: { $gte: parseInt(minPrice) } }, { price: { $lte: parseInt(maxPrice) } }],
    }

    if (ratings.length > 1) {
        filterQuery.avgRating = { $gte: ratings[0], $lte: ratings[1] }
    }
    if (brands.length > 0) {
        filterQuery.brandName = { $in: brands }
    }
    if (color.length > 0) {
        filterQuery.productColor = { $in: color }
    }
    if (size.length > 0) {
        filterQuery.productSize = { $in: size }
    }
    if (category.length > 0) {
        filterQuery.category = { $in: category }
    }

    // console.log(filterQuery);

    Product.find(filterQuery)
        // .select('-photo')
        .populate('category')
        .sort([shortBy])
        .limit(limit)
        .exec((err, products) => {
            if (err || !products) {
                console.error(err);
                return res.status(400).json({
                    error: 'No product found!'
                })
            }
            res.json(products)
        })
}

exports.getRelatedProducts = (req, res) => {
    const { cate } = req.body
    Product.find({ category: cate })
        .populate('category')
        .exec((err, products) => {
            if (err) {
                return res.status(400).json({ message: 'Unable to fetch related products!' })
            }
            res.send(products)
        })

}